// app/api/auth/route.ts
import { NextRequest, NextResponse } from "next/server";
import axios from "axios";
import { BaseUrlNew } from "@/app/lib/constants/urlconstants";

export async function POST(request: NextRequest) {
  try {
    const { action, ...payload } = await request.json();
    
    if (!action) {
      return NextResponse.json(
        { error: "Action parameter is required" },
        { status: 400 }
      );
    }

    switch (action) {
      case "login":
        if (!payload.email || !payload.password) {
          return NextResponse.json(
            { error: "Email and password are required" },
            { status: 400 }
          );
        }

        try {
          const loginResponse = await axios.post(
            `${BaseUrlNew}/auth/login`,
            {
              email: payload.email,
              password: payload.password,
              language: payload.language || "en"
            },
            {
              headers: {
                "Content-Type": "application/json",
              },
            }
          );

          if (loginResponse.data.status === 200) {
            return NextResponse.json({
              success: true,
              data: loginResponse.data,
              token: loginResponse.data.token,
              refreshToken: loginResponse.data.refreshToken,
              user: loginResponse.data.user
            });
          } else {
            return NextResponse.json(
              { 
                error: loginResponse.data.message || "Login failed",
                success: false 
              },
              { status: 400 }
            );
          }
        } catch (error: any) {
          return NextResponse.json(
            { 
              error: error.response?.data?.message || "Login failed",
              success: false 
            },
            { status: error.response?.status || 500 }
          );
        }

      case "register":
        if (!payload.email || !payload.password || !payload.firstName || !payload.lastName) {
          return NextResponse.json(
            { error: "Email, password, first name, and last name are required" },
            { status: 400 }
          );
        }

        try {
          const registerResponse = await axios.post(
            `${BaseUrlNew}/auth/register`,
            payload,
            {
              headers: {
                "Content-Type": "application/json",
              },
            }
          );

          if (registerResponse.data.status === 200 || registerResponse.data.status === 201) {
            return NextResponse.json({
              success: true,
              data: registerResponse.data,
              message: "Registration successful"
            });
          } else {
            return NextResponse.json(
              { 
                error: registerResponse.data.message || "Registration failed",
                success: false 
              },
              { status: 400 }
            );
          }
        } catch (error: any) {
          return NextResponse.json(
            { 
              error: error.response?.data?.message || "Registration failed",
              success: false 
            },
            { status: error.response?.status || 500 }
          );
        }

      default:
        return NextResponse.json(
          { error: "Invalid action type. Use 'login' or 'register'" },
          { status: 400 }
        );
    }
  } catch (error) {
    console.error("Error in authentication API:", error);
    return NextResponse.json(
      {
        error: "Failed to process authentication request",
        success: false
      },
      { status: 500 }
    );
  }
}