import { createAsyncThunk } from "@reduxjs/toolkit";
import axios from "axios";
import { getUserId, getToken, getPromoCode } from "@/app/utils/token";

// Fetch user campaigns
export const fetchEventHubUserCampaigns = createAsyncThunk(
  "campaign/fetchEventHubUserCampaigns",
  async (_, { rejectWithValue }) => {
    try {
      const baseURL = process.env.NEXT_PUBLIC_API_URL_New;
      const token = getToken();
      const userId = getUserId();

      if (!userId) {
        return rejectWithValue("User ID not found");
      }

      if (!token) {
        return rejectWithValue("Authentication token not found");
      }

      const response = await axios.get(
        `${baseURL}/cityhub-campaigns?userId=${userId}`,
        {
          headers: {
            Authorization: `Bearer ${token}`,
            "Content-Type": "application/json",
          },
        }
      );

      if (response.data) {
        return {
          data: response.data,
          message: "Campaigns fetched successfully",
        };
      } else {
        return rejectWithValue("No campaigns data received");
      }
    } catch (error: any) {
      const errorMessage =
        error.response?.data?.message ||
        error.response?.data?.error ||
        "Failed to fetch campaigns";
      return rejectWithValue(errorMessage);
    }
  }
);

// Fetch cities for dropdown
export const fetchCitiesForDropdown = createAsyncThunk(
  "cities/fetchCitiesForDropdown",
  async (_, { rejectWithValue }) => {
    try {
      const baseURL = process.env.NEXT_PUBLIC_API_URL_New;

      const response = await axios.get(
        `${baseURL}/event-hub/get-cities-for-city-hub-campaigns`,
        {
          headers: {
            "Content-Type": "application/json",
          },
        }
      );

      if (response.data) {
        return {
          data: response.data,
          message: "Cities fetched successfully",
        };
      } else {
        return rejectWithValue("No cities data received");
      }
    } catch (error: any) {
      const errorMessage =
        error.response?.data?.message ||
        error.response?.data?.error ||
        "Failed to fetch cities";
      return rejectWithValue(errorMessage);
    }
  }
);

// Postpone campaign
export const postponeCampaign = createAsyncThunk(
  "campaign/postponeEventHubUserCampaign",
  async ({ campaignId, page, limit }: { campaignId: number; page: number; limit: number }, { rejectWithValue }) => {
    try {
      const baseURL = process.env.NEXT_PUBLIC_API_URL_New;
      const token = getToken();
      const userId = getUserId();

      if (!userId) {
        return rejectWithValue("User ID not found");
      }

      if (!token) {
        return rejectWithValue("Authentication token not found");
      }

      const requestData = {
        user_id: userId,
        page: page,
        limit: limit
      };

      const response = await axios.post(
        `${baseURL}/cityhub-campaigns/${campaignId}/postpone`,
        requestData,
        {
          headers: {
            Authorization: `Bearer ${token}`,
            "Content-Type": "application/json",
          },
        }
      );

      if (response.data) {
        return {
          data: response.data,
          message: "Event Hub Campaign postponed successfully",
          campaignId: campaignId
        };
      } else {
        return rejectWithValue("No response data received");
      }
    } catch (error: any) {
      const errorMessage =
        error.response?.data?.message ||
        error.response?.data?.error ||
        "Failed to postpone campaign";
      return rejectWithValue(errorMessage);
    }
  }
);
export const deleteEventHubCampaign = createAsyncThunk(
  "campaign/deleteEventHubUserCampaign",
  async (campaignId: number, { rejectWithValue }) => {
    try {
      const baseURL = process.env.NEXT_PUBLIC_API_URL_New;
      const token = getToken();
      const userId = getUserId();

      if (!userId) {
        return rejectWithValue("User ID not found");
      }

      if (!token) {
        return rejectWithValue("Authentication token not found");
      }

      const requestData = {
        user_id: userId,
        page: 1,
        limit: 100
      };

      const response = await axios.delete(
        `${baseURL}/cityhub-campaigns/${campaignId}`,
        {
          headers: {
            'Authorization': `Bearer ${token}`,
            'Content-Type': 'text/plain',
          },
          data: requestData
        }
      );

      if (response.data) {
        return {
          data: response.data,
          message: "Campaign deleted successfully",
          campaignId: campaignId
        };
      } else {
        return rejectWithValue("No response data received");
      }
    } catch (error: any) {
      const errorMessage =
        error.response?.data?.message ||
        error.response?.data?.error ||
        "Failed to delete campaign";
      return rejectWithValue(errorMessage);
    }
  }
);