import { useState, useEffect } from "react";
import {
  getSepData,
  getCityHubData,
  getCategoryData,
  getSummaryData,
  getSepTableData,
  getEventHubTableData,
  getCategoryTableData,
  getSidebarData,
  getSettingsData,
} from "@/app/lib/json/data";
type DataType =
  | "sepData"
  | "cityHubData"
  | "categoryData"
  | "summaryData"
  | "sepTableData"
  | "eventHubTableData"
  | "categoryTableData"
  | "sidebarData"
  | "settingsData";
// Base hook interface
interface UseDataReturn<T> {
  data: T | null;
  loading: boolean;
  error: string | null;
  refetch: () => void;
}

export const useData = <T = any>(
  fetchFunction: () => Promise<T>
): UseDataReturn<T> => {
  const [data, setData] = useState<T | null>(null);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);

  const fetchData = async () => {
    try {
      setLoading(true);
      setError(null);
      const result = await fetchFunction();
      setData(result);
    } catch (err) {
      setError(err instanceof Error ? err.message : "Failed to load data");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData();
  }, [fetchFunction]);

  return { data, loading, error, refetch: fetchData };
};

export const useSepData = () => useData(getSepData);
export const useCityHubData = () => useData(getCityHubData);
export const useCategoryData = () => useData(getCategoryData);
export const useSummaryData = () => useData(getSummaryData);
export const useSepTableData = () => useData(getSepTableData);
export const useEventHubTableData = () => useData(getEventHubTableData);
export const useCategoryTableData = () => useData(getCategoryTableData);
export const useSidebarData = () => useData(getSidebarData);
export const useSettingsData = () => useData(getSettingsData);

// Note: Auth-related hooks (useLoginData, useRegisterData, etc.) have been removed.
// Translations are now defined inline in their respective components.

export const useDataByType = (dataType: DataType) => {
  const fetchFunctions = {
    sepData: getSepData,
    cityHubData: getCityHubData,
    categoryData: getCategoryData,
    summaryData: getSummaryData,
    sepTableData: getSepTableData,
    eventHubTableData: getEventHubTableData,
    categoryTableData: getCategoryTableData,
    sidebarData: getSidebarData,
    settingsData: getSettingsData,
  };

  return useData(fetchFunctions[dataType]);
};
