type DataType =
  | "sepData"
  | "cityHubData"
  | "categoryData"
  | "summaryData"
  | "sepTableData"
  | "eventHubTableData"
  | "categoryTableData"
  | "sidebarData"
  | "loginData"
  | "registerData"
  | "forgetPasswordData"
  | "verifyEmailData"
  | "verifyCodeData"
  | "resetPasswordData"
  | "settingsData";

interface DataSources {
  sepData: () => Promise<any>;
  cityHubData: () => Promise<any>;
  categoryData: () => Promise<any>;
  summaryData: () => Promise<any>;
  sepTableData: () => Promise<any>;
  eventHubTableData: () => Promise<any>;
  categoryTableData: () => Promise<any>;
  sidebarData: () => Promise<any>;
  loginData: () => Promise<any>;
  registerData: () => Promise<any>;
  forgetPasswordData: () => Promise<any>;
  verifyEmailData: () => Promise<any>;
  verifyCodeData: () => Promise<any>;
  resetPasswordData: () => Promise<any>;
  settingsData: () => Promise<any>;
}

const dataSources: DataSources = {
  sepData: () =>
    import("./selfserveportal.json").then((module) => module.default),
  cityHubData: () =>
    import("./cityhubpages.json").then((module) => module.default),
  categoryData: () =>
    import("./category.json").then((module) => module.default),
  summaryData: () => import("./summary.json").then((module) => module.default),
  sepTableData: () =>
    import("./sepTableJson.json").then((module) => module.default),
  eventHubTableData: () =>
    import("./eventHubTableJson.json").then((module) => module.default),
  categoryTableData: () =>
    import("./categoryTableJson.json").then((module) => module.default),
  sidebarData: () => import("./sidebar.json").then((module) => module.default),
  loginData: () => import("./login.json").then((module) => module.default),
  registerData: () =>
    import("./register.json").then((module) => module.default),
  forgetPasswordData: () =>
    import("./forgetpassword.json").then((module) => module.default),
  verifyEmailData: () =>
    import("./verifyEmail.json").then((module) => module.default),
  verifyCodeData: () =>
    import("./verifyCode.json").then((module) => module.default),
  resetPasswordData: () =>
    import("./resetPassword.json").then((module) => module.default),
  settingsData: () =>
    import("./settings.json").then((module) => module.default),
};

export const getData = async (dataType: DataType): Promise<any> => {
  if (!dataSources[dataType]) {
    throw new Error(
      `Data type '${dataType}' not found. Available types: ${Object.keys(
        dataSources
      ).join(", ")}`
    );
  }

  try {
    return await dataSources[dataType]();
  } catch (error) {
    console.error(`Error loading ${dataType}:`, error);
    throw new Error(`Failed to load ${dataType} data`);
  }
};

// Individual exports
export const getSepData = (): Promise<any> => getData("sepData");
export const getCityHubData = (): Promise<any> => getData("cityHubData");
export const getCategoryData = (): Promise<any> => getData("categoryData");
export const getSummaryData = (): Promise<any> => getData("summaryData");
export const getSepTableData = (): Promise<any> => getData("sepTableData");
export const getEventHubTableData = (): Promise<any> =>
  getData("eventHubTableData");
export const getCategoryTableData = (): Promise<any> =>
  getData("categoryTableData");
export const getSidebarData = (): Promise<any> => getData("sidebarData");
export const getLoginData = (): Promise<any> => getData("loginData");
export const getRegisterData = (): Promise<any> => getData("registerData");
export const getForgetPasswordData = (): Promise<any> =>
  getData("forgetPasswordData");
export const getVerifyEmailData = (): Promise<any> =>
  getData("verifyEmailData");
export const getVerifyCodeData = (): Promise<any> => getData("verifyCodeData");
export const getResetPasswordData = (): Promise<any> =>
  getData("resetPasswordData");
export const getSettingsData = (): Promise<any> => getData("settingsData");

export default dataSources;
