import { useState, useEffect } from "react";
import {
  getSepData,
  getCityHubData,
  getCategoryData,
  getSummaryData,
  getSepTableData,
  getEventHubTableData,
  getCategoryTableData,
  getSidebarData,
  getLoginData,
  getRegisterData,
  getForgetPasswordData,
  getVerifyEmailData,
  getVerifyCodeData,
  getResetPasswordData,
  getSettingsData,
} from "@/app/lib/json/data";
type DataType =
  | "sepData"
  | "cityHubData"
  | "categoryData"
  | "summaryData"
  | "sepTableData"
  | "eventHubTableData"
  | "categoryTableData"
  | "sidebarData"
  | "loginData"
  | "registerData"
  | "forgetPasswordData"
  | "verifyEmailData"
  | "verifyCodeData"
  | "resetPasswordData"
  | "settingsData";
// Base hook interface
interface UseDataReturn<T> {
  data: T | null;
  loading: boolean;
  error: string | null;
  refetch: () => void;
}

export const useData = <T = any>(
  fetchFunction: () => Promise<T>
): UseDataReturn<T> => {
  const [data, setData] = useState<T | null>(null);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<string | null>(null);

  const fetchData = async () => {
    try {
      setLoading(true);
      setError(null);
      const result = await fetchFunction();
      setData(result);
    } catch (err) {
      setError(err instanceof Error ? err.message : "Failed to load data");
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData();
  }, [fetchFunction]);

  return { data, loading, error, refetch: fetchData };
};

export const useSepData = () => useData(getSepData);
export const useCityHubData = () => useData(getCityHubData);
export const useCategoryData = () => useData(getCategoryData);
export const useSummaryData = () => useData(getSummaryData);
export const useSepTableData = () => useData(getSepTableData);
export const useEventHubTableData = () => useData(getEventHubTableData);
export const useCategoryTableData = () => useData(getCategoryTableData);
export const useSidebarData = () => useData(getSidebarData);
export const useLoginData = () => useData(getLoginData);
export const useRegisterData = () => useData(getRegisterData);
export const useForgetPasswordData = () => useData(getForgetPasswordData);
export const useVerifyEmailData = () => useData(getVerifyEmailData);
export const useVerifyCodeData = () => useData(getVerifyCodeData);
export const useResetPasswordData = () => useData(getResetPasswordData);
export const useSettingsData = () => useData(getSettingsData);

export const useDataByType = (dataType: DataType) => {
  const fetchFunctions = {
    sepData: getSepData,
    cityHubData: getCityHubData,
    categoryData: getCategoryData,
    summaryData: getSummaryData,
    sepTableData: getSepTableData,
    eventHubTableData: getEventHubTableData,
    categoryTableData: getCategoryTableData,
    sidebarData: getSidebarData,
    loginData: getLoginData,
    registerData: getRegisterData,
    forgetPasswordData: getForgetPasswordData,
    verifyEmailData: getVerifyEmailData,
    verifyCodeData: getVerifyCodeData,
    resetPasswordData: getResetPasswordData,
    settingsData: getSettingsData,
  };

  return useData(fetchFunctions[dataType]);
};
