"use client";
import React, { useEffect, useState, useRef } from "react";
import "./table.css";
import Swal from "sweetalert2";
import { useAppDispatch, useAppSelector } from "@/app/hooks/useDispatch";
import { useRouter } from "next/navigation";
// import { calculatePaidDays } from "@/app/utils/datehelpers";
import InvoiceModal from "./InvoiceView";
import {
  fetchEventHubUserCampaigns,
  postponeCampaign,
  deleteEventHubCampaign,
} from "@/app/redux/services/main/campaigns/evenhubcampaigns";
import { useEventHubTableData } from "@/app/hooks/useData";

const Adverts = () => {
  const dispatch = useAppDispatch();
  const router = useRouter();
  const { data: translations } = useEventHubTableData();
  const t = translations?.Section || {};
  const invoiceModalRef = useRef<HTMLDivElement>(null);
  const [invoiceModal, setInvoiceModal] = useState({
    isOpen: false,
    invoiceData: null as any,
  });
  const [statusFilter, setStatusFilter] = useState<
    "all" | "active" | "postponed"
  >("all");

  const { eventHubCompaigns, loadingCommuneHubs } = useAppSelector(
    (state) => state.eventhub
  );
  const [postponedCampaigns, setPostponedCampaigns] = useState<Set<number>>(
    new Set()
  );

  const [openDropdownId, setOpenDropdownId] = useState<number | null>(null);
  const [mediaModal, setMediaModal] = useState({
    isOpen: false,
    type: "",
    url: "",
  });

  // Filter campaigns based on status
  const filteredCampaigns = eventHubCompaigns?.filter((campaign: any) => {
    if (statusFilter === "all") return true;
    if (statusFilter === "active") return campaign.status !== "postpone";
    if (statusFilter === "postponed") return campaign.status === "postpone";
    return true;
  });

  const togglePostpone = async (campaignId: number) => {
    try {
      await dispatch(
        postponeCampaign({
          campaignId,
          page: 1,
          limit: 100,
        })
      ).unwrap();

      dispatch(fetchEventHubUserCampaigns());

      setPostponedCampaigns((prev) => {
        const newSet = new Set(prev);
        if (newSet.has(campaignId)) {
          newSet.delete(campaignId);
        } else {
          newSet.add(campaignId);
        }
        return newSet;
      });
    } catch (error: any) {
      console.error("Failed to postpone campaign:", error);
      Swal.fire({
        title: t.SweetAlert?.Postpone_Error_Title || "Error!",
        text:
          error ||
          t.SweetAlert?.Postpone_Error_Text ||
          "Failed to postpone campaign",
        icon: "error",
        confirmButtonColor: "#d33",
      });
    }
  };

  useEffect(() => {
    dispatch(fetchEventHubUserCampaigns());
  }, [dispatch]);

  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      closeInvoiceModal();
      if (
        invoiceModalRef.current &&
        !invoiceModalRef.current.contains(event.target as Node) &&
        invoiceModal.isOpen
      ) {
        closeInvoiceModal();
      }
    };

    if (invoiceModal.isOpen) {
      document.addEventListener("mousedown", handleClickOutside);
      document.body.style.overflow = "hidden";
    }

    return () => {
      document.removeEventListener("mousedown", handleClickOutside);
      document.body.style.overflow = "unset";
    };
  }, [invoiceModal.isOpen]);

  const handleDownloadInvoice = async (campaign: any) => {
    try {
      const invoiceData = {
        campaignId: campaign.id,
        campaignTitle: campaign.title,
        createdAt: campaign.created_at,
        organization: campaign.user?.organization,
        name: campaign.user?.name,
        address: campaign.user?.address,
        userId: campaign.user_id,
        payment: campaign.payments?.[0] || {},
        totalAmount: campaign.payments?.[0]?.final_amount || "0.00",
        // Media information
        mediaType: getMediaTypeFromPath(campaign.media_path),
        mediaFilePath: campaign.media_path,
        // Additional campaign details
        status: campaign.status,
        userNotes: campaign.user_notes,
        adminNotes: campaign.admin_notes,
        discount: campaign.payments?.[0]?.discount || "0",
        // EventHub specific data
        communeName: campaign.cityHub?.commune_name,
        duration: campaign.duration,
        startDate: campaign.start_date,
        endDate: campaign.end_date,
      };

      setInvoiceModal({
        isOpen: true,
        invoiceData,
      });
    } catch (error) {
      console.error("Error downloading invoice:", error);
    }
  };

  const closeInvoiceModal = () => {
    setInvoiceModal({
      isOpen: false,
      invoiceData: null,
    });
  };

  useEffect(() => {
    if (openDropdownId !== null) {
      const handleClickOutside = () => {
        setOpenDropdownId(null);
      };
      document.addEventListener("click", handleClickOutside);
      return () => {
        document.removeEventListener("click", handleClickOutside);
      };
    }
  }, [openDropdownId]);

  // Function to get file extension and determine media type
  const getMediaTypeFromPath = (filePath: string) => {
    if (!filePath) return "unknown";

    const extension = filePath.split(".").pop()?.toLowerCase();

    // Video extensions
    const videoExtensions = ["mp4", "mov", "avi", "wmv", "flv", "webm", "mkv"];
    // Rich text/document extensions
    const richTextExtensions = ["rtf", "doc", "docx", "txt", "pdf"];
    // Image extensions
    const imageExtensions = ["jpg", "jpeg", "png", "gif", "bmp", "webp", "svg"];

    if (videoExtensions.includes(extension || "")) {
      return "video";
    } else if (richTextExtensions.includes(extension || "")) {
      return "richtext";
    } else if (imageExtensions.includes(extension || "")) {
      return "image";
    } else {
      return "unknown";
    }
  };

  const handleDeleteCampaign = async (campaignId: number) => {
    const result = await Swal.fire({
      title: t.SweetAlert?.Delete_Confirm_Title || "Are you sure?",
      text:
        t.SweetAlert?.Delete_Confirm_Text ||
        "You won't be able to revert this!",
      icon: "warning",
      showCancelButton: true,
      confirmButtonColor: "#d33",
      cancelButtonColor: "#3085d6",
      confirmButtonText:
        t.SweetAlert?.Delete_Confirm_Button || "Yes, delete it!",
      cancelButtonText: t.SweetAlert?.Delete_Cancel_Button || "Cancel",
      reverseButtons: true,
      customClass: {
        confirmButton: "swal2-confirm",
        cancelButton: "swal2-cancel",
      },
    });

    if (result.isConfirmed) {
      try {
        // Pass only campaignId since the thunk expects a single number parameter
        await dispatch(deleteEventHubCampaign(campaignId)).unwrap();

        // Refresh the campaigns list after deletion
        dispatch(fetchEventHubUserCampaigns());

        Swal.fire({
          title: t.SweetAlert?.Delete_Success_Title || "Deleted!",
          text:
            t.SweetAlert?.Delete_Success_Text ||
            "Your campaign has been deleted.",
          icon: "success",
          confirmButtonColor: "#3085d6",
          timer: 2000,
          showConfirmButton: false,
        });
      } catch (error: any) {
        console.error("Failed to delete campaign:", error);
        Swal.fire({
          title: t.SweetAlert?.Delete_Error_Title || "Error!",
          text:
            error ||
            t.SweetAlert?.Delete_Error_Text ||
            "Failed to delete campaign",
          icon: "error",
          confirmButtonColor: "#d33",
        });
      }
    }
  };

  const handleMediaPreviewClick = (campaign: any) => {
    const mediaType = getMediaTypeFromPath(campaign.media_path);
    const fileUrl = `${process.env.AWS_BUCKET}${campaign.media_path}`;

    setMediaModal({
      isOpen: true,
      type: mediaType,
      url: fileUrl,
    });
  };
  // Function to handle city click and redirect to EventHub page
  const handleCityClick = (communeSlug: string) => {
    if (communeSlug) {
      window.open(
        `https://stagging.eventx.fr/events/eventhub/${communeSlug}`,
        "_blank"
      );
    }
  };
  const closeMediaModal = () => {
    setMediaModal({ isOpen: false, type: "", url: "" });
  };

  const renderMediaPreview = (campaign: any) => {
    const mediaType = getMediaTypeFromPath(campaign.media_path);
    const filePath = campaign.media_path;
    const fileUrl = `${process.env.AWS_BUCKET}${filePath}`;

    switch (mediaType) {
      case "video":
        return (
          <div
            className="relative w-12 h-12 bg-gray-200 rounded flex items-center justify-center cursor-pointer hover:bg-gray-300 transition-colors"
            onClick={() => handleMediaPreviewClick(campaign)}
            title={t.Media_Modal?.Click_To_View_Video || "Click to view video"}
          >
            <svg
              className="w-6 h-6 text-gray-600"
              fill="currentColor"
              viewBox="0 0 24 24"
            >
              <path d="M8 5v14l11-7z" />
            </svg>
            <div className="absolute bottom-0 right-0 bg-black bg-opacity-70 text-white text-xs px-1 rounded-tl">
              MP4
            </div>
          </div>
        );

      case "richtext":
        return (
          <div
            className="relative w-12 h-12 bg-blue-100 rounded flex items-center justify-center cursor-pointer hover:bg-blue-200 transition-colors"
            onClick={() => handleMediaPreviewClick(campaign)}
            title={
              t.Media_Modal?.Click_To_View_Document || "Click to view document"
            }
          >
            <svg
              className="w-6 h-6 text-blue-600"
              fill="currentColor"
              viewBox="0 0 24 24"
            >
              <path d="M14 2H6c-1.1 0-1.99.9-1.99 2L4 20c0 1.1.89 2 1.99 2H18c1.1 0 2-.9 2-2V8l-6-6zm2 16H8v-2h8v2zm0-4H8v-2h8v2zm-3-5V3.5L18.5 9H13z" />
            </svg>
            <div className="absolute bottom-0 right-0 bg-blue-600 text-white text-xs px-1 rounded-tl">
              DOC
            </div>
          </div>
        );

      case "image":
      default:
        return (
          <div
            className="cursor-pointer"
            onClick={() => handleMediaPreviewClick(campaign)}
            title={t.Media_Modal?.Click_To_View_Image || "Click to view image"}
          >
            <img
              src={fileUrl || "/event-table-img.png"}
              alt={t.Alt_Text?.Ad || "Ad"}
              className="w-12 h-12 object-cover rounded hover:opacity-90 transition-opacity"
              onError={(e) => {
                e.currentTarget.style.display = "none";
                const parent = e.currentTarget.parentElement;
                if (parent) {
                  const fallback = document.createElement("div");
                  fallback.className =
                    "w-12 h-12 bg-gray-200 rounded flex items-center justify-center cursor-pointer hover:bg-gray-300 transition-colors";
                  fallback.innerHTML = `
                    <svg class="w-6 h-6 text-gray-600" fill="currentColor" viewBox="0 0 24 24">
                      <path d="M21 19V5c0-1.1-.9-2-2-2H5c-1.1 0-2 .9-2 2v14c0 1.1.9 2 2 2h14c1.1 0 2-.9 2-2zM8.5 13.5l2.5 3.01L14.5 12l4.5 6H5l3.5-4.5z"/>
                    </svg>
                  `;
                  fallback.onclick = () => handleMediaPreviewClick(campaign);
                  parent.appendChild(fallback);
                }
              }}
            />
          </div>
        );
    }
  };

  const renderMediaModalContent = () => {
    switch (mediaModal.type) {
      case "video":
        return (
          <div className="w-full max-w-4xl mx-auto">
            <video
              controls
              autoPlay
              className="w-full h-auto max-h-[80vh] rounded-lg"
            >
              <source src={mediaModal.url} type="video/mp4" />
              {t.Common?.Video_Not_Supported ||
                "Your browser does not support the video tag."}
            </video>
          </div>
        );

      case "richtext":
        return (
          <div className="w-full max-w-4xl mx-auto bg-white rounded-lg p-6">
            <div className="text-center mb-4">
              <svg
                className="w-16 h-16 text-blue-500 mx-auto mb-4"
                fill="currentColor"
                viewBox="0 0 24 24"
              >
                <path d="M14 2H6c-1.1 0-1.99.9-1.99 2L4 20c0 1.1.89 2 1.99 2H18c1.1 0 2-.9 2-2V8l-6-6zm2 16H8v-2h8v2zm0-4H8v-2h8v2zm-3-5V3.5L18.5 9H13z" />
              </svg>
              <h3 className="text-xl font-semibold text-gray-800">
                {t.Media_Modal?.Document || "Document File"}
              </h3>
              <p className="text-gray-600 mt-2">
                {t.Media_Modal?.Document_Description ||
                  "This is a document file that can be downloaded."}
              </p>
            </div>
          </div>
        );

      case "image":
        return (
          <div className="w-full max-w-4xl mx-auto">
            <img
              src={mediaModal.url}
              alt={t.Alt_Text?.Ad_Preview || "Ad preview"}
              className="w-full h-auto max-h-[80vh] object-contain rounded-lg"
            />
          </div>
        );

      default:
        return (
          <div className="text-center p-8">
            <p className="text-lg text-gray-600">
              {t.Media_Modal?.Unable_To_Preview ||
                "Unable to preview this file type."}
            </p>
          </div>
        );
    }
  };

  // Format date for display
  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString("en-US", {
      year: "numeric",
      month: "short",
      day: "numeric",
    });
  };

  // Get status display text and color
  const getStatusDisplay = (status: string) => {
    switch (status) {
      case "0":
        return { text: "Active", color: "text-green-600" };
      case "1":
        return { text: "Inactive", color: "text-red-600" };
      case "postpone":
        return { text: "Postponed", color: "text-orange-600" };
      default:
        return { text: status, color: "text-gray-600" };
    }
  };

  return (
    <>
      <h2 className="aileron-heavy text-[1.5rem] sm:text-[2rem] lg:text-[3.1rem] leading-none">
        {t.Title || "MY EVENTHUB CAMPAIGNS"}
      </h2>
      <p className="aileron-semibold text-[1.2rem] sm:text-[1.4rem] lg:text-[1.6rem] leading-none mt-2">
        {t.Subtitle || "By City Hub"}
      </p>

      {mediaModal.isOpen && (
        <div className="fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center z-50 p-4">
          <div className="relative bg-white rounded-lg max-w-4xl w-full max-h-[90vh] overflow-auto">
            <button
              onClick={closeMediaModal}
              className="absolute top-4 right-4 text-gray-500 hover:text-gray-700 z-10 bg-white rounded-full p-2"
            >
              <svg
                className="w-6 h-6"
                fill="none"
                stroke="currentColor"
                viewBox="0 0 24 24"
              >
                <path
                  strokeLinecap="round"
                  strokeLinejoin="round"
                  strokeWidth={2}
                  d="M6 18L18 6M6 6l12 12"
                />
              </svg>
            </button>
            <div className="p-6">{renderMediaModalContent()}</div>
          </div>
        </div>
      )}

      <div
        style={{ boxShadow: "rgba(0, 0, 0, 0.24) 0px 3px 8px" }}
        className="bg-[#F2F2F2] rounded-lg p-3 sm:p-4 lg:p-5 flex flex-col sm:flex-row items-stretch sm:items-center justify-between gap-4 sm:gap-6 lg:gap-8 mt-4 sm:mt-5 mb-8 sm:mb-10 lg:mb-12"
      >
        <select
          className="bg-white p-2.5 sm:p-3 w-full sm:w-2/6 rounded-lg border-[0.25px] border-grey text-[0.95rem] sm:text-[1rem] lg:text-[1.1rem] aileron-semibold"
          name=""
          id=""
          value={statusFilter}
          onChange={(e) =>
            setStatusFilter(e.target.value as "all" | "active" | "postponed")
          }
        >
          <option value="active">{t.Filters?.Filter_2 || "Online"}</option>
          <option value="postponed">
            {t.Filters?.Filter_3 || "Postponed"}
          </option>
          <option value="all">{t.Filters?.Filter_1 || "All"}</option>
        </select>

        <button
          onClick={() => router.push("/campaigns/createHubPageAdvert")}
          className="bg-gradient-to-r from-[#C1272D] to-[#a02025] hover:from-[#a02025] hover:to-[#8a1b20] text-white px-4 sm:px-6 lg:px-8 py-2.5 sm:py-3 rounded-lg text-[0.95rem] sm:text-[1rem] lg:text-[1.1rem] aileron-bold transition-all duration-300 flex items-center justify-center gap-2 sm:gap-3 shadow-lg hover:shadow-xl transform hover:scale-105 w-full sm:w-auto"
        >
          <svg
            width="18"
            height="18"
            className="sm:w-5 sm:h-5"
            viewBox="0 0 24 24"
            fill="none"
            xmlns="http://www.w3.org/2000/svg"
          >
            <path
              d="M12 5V19M5 12H19"
              stroke="currentColor"
              strokeWidth="2"
              strokeLinecap="round"
              strokeLinejoin="round"
            />
          </svg>
          <span>
            {t.Buttons?.Add_New_Campaign || "Add New EventHub Campaign"}
          </span>
        </button>
      </div>

      {/* Table Container */}
      <div className="w-full mt-4 sm:mt-5">
        {invoiceModal.isOpen && (
          <div
            className="fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center z-50 p-4"
            onClick={closeInvoiceModal}
          >
            <div
              className="relative bg-white rounded-lg max-w-4xl w-full max-h-[90vh] overflow-auto"
              onClick={(e) => e.stopPropagation()}
            >
              <button
                onClick={closeInvoiceModal}
                className="absolute top-4 right-4 text-gray-500 hover:text-gray-700 z-10 bg-white rounded-full p-2"
              >
                <svg
                  className="w-6 h-6"
                  fill="none"
                  stroke="currentColor"
                  viewBox="0 0 24 24"
                >
                  <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M6 18L18 6M6 6l12 12"
                  />
                </svg>
              </button>
              <InvoiceModal
                isOpen={invoiceModal.isOpen}
                onClose={closeInvoiceModal}
                invoiceData={invoiceModal.invoiceData}
              />
            </div>
          </div>
        )}

        <div className="table-card-content-parentdesktop-view">
          <div className="overflow-hidden sm:rounded-md">
            <table className="min-w-full bg-white">
              <thead>
                <tr className="text-xs leading-4 tracking-wider border-b">
                  <th className="px-3 xl:px-6 py-3 text-center aileron-bold">
                    <p>{t.Table_Columns?.Column_1 || "Title"}</p>
                  </th>
                  <th className="px-3 xl:px-6 py-3 text-center aileron-bold">
                    <p>{t.Table_Columns?.Column_2 || "Media"}</p>
                  </th>
                  <th className="px-3 xl:px-6 py-3 text-center aileron-bold">
                    <p>{t.Table_Columns?.Column_3 || "City"}</p>
                  </th>

                  {/* Actions at position 4 like SEP campaigns */}
                  <th className="px-3 xl:px-6 py-3 text-center aileron-bold">
                    <p>{t.Table_Columns?.Column_4 || "Actions"}</p>
                  </th>
                  <th className="px-3 xl:px-6 py-3 text-center aileron-bold">
                    <p>{t.Table_Columns?.Column_5 || "Postpone"}</p>
                  </th>
                  <th className="px-2 xl:px-6 py-3 text-center aileron-bold">
                    <p>{t.Table_Columns?.Column_6 || "Bill.pdf"}</p>
                  </th>
                  {/* Refund and Notes eyes at the end like SEP campaigns */}
                  <th className="px-3 xl:px-6 py-3 text-center aileron-bold">
                    <p>{t.Table_Columns?.Column_7 || "Refunded"}</p>
                  </th>
                  <th className="px-3 xl:px-6 py-3 text-center aileron-bold">
                    <p>{t.Table_Columns?.Column_8 || "Cost€"}</p>
                  </th>
                  <th className="px-3 xl:px-6 py-3 text-center aileron-bold">
                    <p>{t.Table_Columns?.Column_9 || "Notes"}</p>
                  </th>
                </tr>
              </thead>
              <tbody>
                {loadingCommuneHubs ? (
                  <tr>
                    <td colSpan={11} className="px-6 py-8 text-center">
                      <p className="aileron-semibold text-[1.2rem] sm:text-[1.4rem] text-gray-500">
                        {t.Loading?.Loading_Campaigns || "Loading campaigns"}
                      </p>
                    </td>
                  </tr>
                ) : filteredCampaigns?.length === 0 ? (
                  <tr>
                    <td colSpan={11} className="px-6 py-8 text-center">
                      <p className="aileron-semibold text-[1.2rem] sm:text-[1.4rem] text-gray-500">
                        {t.Loading?.No_Campaigns_Found || "No campaigns found"}
                      </p>
                    </td>
                  </tr>
                ) : (
                  filteredCampaigns?.map((campaign: any) => {
                    const statusDisplay = getStatusDisplay(campaign.status);
                    return (
                      <tr
                        key={campaign.id}
                        className="bg-[#fafdff] border-b hover:bg-gray-50"
                      >
                        <td className="px-3 xl:px-6 py-4 whitespace-no-wrap">
                          <div className="flex items-center justify-center">
                            <p className="avenir-roman text-sm">
                              {campaign.title}
                            </p>
                          </div>
                        </td>
                        <td className="px-3 xl:px-6 py-4 whitespace-no-wrap">
                          <div className="flex items-center justify-center">
                            {renderMediaPreview(campaign)}
                          </div>
                        </td>
                        <td className="px-3 xl:px-6 py-4 whitespace-no-wrap">
                          <div className="flex items-center justify-center">
                            <p className="text-sm">
                              {campaign.cityHub?.commune_name ||
                                t.Common?.N_A ||
                                "N/A"}
                            </p>
                          </div>
                        </td>

                        {/* Actions at position 4 */}
                        <td className="px-3 xl:px-6 py-4 whitespace-no-wrap">
                          <div className="flex items-center justify-center gap-2">
                            <img
                              className="cursor-pointer w-5 h-5"
                              src="/blue-eye-icon.png"
                              alt={t.Alt_Text?.View || "View"}
                              onClick={() =>
                                handleCityClick(campaign.cityHub?.commune_slug)
                              }
                            />
                            <img
                              className="cursor-pointer w-5 h-5"
                              src="/delete-icon.png"
                              alt={t.Alt_Text?.Delete || "Delete"}
                              onClick={() => handleDeleteCampaign(campaign.id)}
                            />
                          </div>
                        </td>
                        <td className="px-3 xl:px-6 py-4 whitespace-no-wrap">
                          <div className="flex items-center justify-center">
                            <label className="relative inline-block w-10 h-5 cursor-pointer">
                              <input
                                type="checkbox"
                                checked={campaign.status === "postpone"}
                                onChange={() => togglePostpone(campaign.id)}
                                className="hidden"
                              />
                              <div
                                className={`absolute inset-0 border-2 rounded-full transition-all duration-200 ${
                                  campaign.status === "postpone"
                                    ? "border-green-500 bg-green-100"
                                    : "border-black bg-white"
                                }`}
                              ></div>
                              <div
                                className={`absolute top-1/2 -translate-y-1/2 w-3.5 h-3.5 rounded-full transition-all duration-200 ${
                                  campaign.status === "postpone"
                                    ? "left-[calc(100%-1.1rem)] bg-green-500"
                                    : "left-[0.2rem] bg-black"
                                }`}
                              ></div>
                            </label>
                          </div>
                        </td>
                        <td className="px-2 xl:px-6 py-4 whitespace-no-wrap">
                          <div
                            className="flex items-center justify-center"
                            onClick={() => handleDownloadInvoice(campaign)}
                          >
                            <img
                              className="cursor-pointer w-5 h-5"
                              src="/blue-eye-icon.png"
                              alt={t.Alt_Text?.Invoice || "Invoice"}
                            />
                          </div>
                        </td>
                        {/* Refund and Notes eyes at the end */}
                        <td className="px-3 xl:px-6 py-4 whitespace-no-wrap">
                          <div className="flex items-center justify-center">
                            <img
                              className="cursor-pointer w-5 h-5"
                              src="/red-circle.png"
                              alt={t.Alt_Text?.Refund || "Refund"}
                            />
                          </div>
                        </td>
                        <td className="px-3 xl:px-6 py-4 whitespace-no-wrap">
                          <div className="flex items-center justify-center">
                            <p className="avenir-roman text-sm">
                              {campaign.payments?.[0]?.final_amount || "0"}
                            </p>
                          </div>
                        </td>
                        <td className="px-3 xl:px-6 py-4 whitespace-no-wrap">
                          <div className="flex items-center justify-center">
                            <img
                              className="cursor-pointer w-5 h-5"
                              src="/blue-eye-icon.png"
                              alt={t.Alt_Text?.Notes || "Notes"}
                            />
                          </div>
                        </td>
                      </tr>
                    );
                  })
                )}
              </tbody>
            </table>
          </div>
        </div>

        {/* Mobile/Tablet Card View */}
        <div className="space-y-4 table-card-content-parentmobile-view">
          {loadingCommuneHubs ? (
            <div className="bg-white rounded-lg shadow p-6 text-center">
              <p className="aileron-semibold text-[1.2rem] text-gray-500">
                {t.Loading?.Loading_Campaigns || "Loading campaigns"}
              </p>
            </div>
          ) : filteredCampaigns?.length === 0 ? (
            <div className="bg-white rounded-lg shadow p-6 text-center">
              <p className="aileron-semibold text-[1.2rem] text-gray-500">
                {t.Loading?.No_Campaigns_Found || "No campaigns found"}
              </p>
            </div>
          ) : (
            filteredCampaigns?.map((campaign: any) => {
              const statusDisplay = getStatusDisplay(campaign.status);
              return (
                <div
                  key={campaign.id}
                  className="bg-[#fafdff] rounded-lg shadow-md p-4 sm:p-5 space-y-3"
                >
                  {/* Row 1: Title, City, Status */}
                  <div className="flex items-center justify-between border-b pb-3">
                    <div className="flex-1">
                      <p className="text-xs text-gray-500 aileron-bold mb-1">
                        {t.Mobile_View?.Title || "Title"}
                      </p>
                      <p className="avenir-roman text-sm font-semibold">
                        {campaign.title}
                      </p>
                    </div>
                    <div className="flex-1 text-center">
                      <p className="text-xs text-gray-500 aileron-bold mb-1">
                        {t.Mobile_View?.Media || "Media"}
                      </p>
                      <div className="flex justify-center">
                        {renderMediaPreview(campaign)}
                      </div>
                    </div>
                    <div className="flex-1 text-center">
                      <p className="text-xs text-gray-500 aileron-bold mb-1">
                        {t.Mobile_View?.City || "City"}
                      </p>
                      <p className="text-sm">
                        {campaign.cityHub?.commune_name ||
                          t.Common?.N_A ||
                          "N/A"}
                      </p>
                    </div>
                  </div>

                  {/* Row 2: Duration and Media */}
                  {/* <div className="flex items-center justify-between border-b pb-3">
          
            <div className="flex-1 text-center">
              <p className="text-xs text-gray-500 aileron-bold mb-1">
                Cost€
              </p>
              <p className="avenir-roman text-sm font-semibold">
                {campaign.payments?.[0]?.final_amount || "0"}
              </p>
            </div>
          </div> */}

                  {/* Row 3: Actions, Postpone, Bill */}
                  <div className="flex items-center justify-between border-b pb-3">
                    <div className="flex-1">
                      <p className="text-xs text-gray-500 aileron-bold mb-1">
                        {t.Mobile_View?.Actions || "Actions"}
                      </p>
                      <div className="flex items-center justify-center gap-3">
                        <img
                          className="cursor-pointer w-6 h-6"
                          src="/blue-eye-icon.png"
                          alt={t.Alt_Text?.View || "View"}
                          onClick={() =>
                            handleCityClick(campaign.cityHub?.commune_slug)
                          }
                        />
                        <img
                          className="cursor-pointer w-6 h-6"
                          src="/delete-icon.png"
                          alt={t.Alt_Text?.Delete || "Delete"}
                          onClick={() => handleDeleteCampaign(campaign.id)}
                        />
                      </div>
                    </div>
                    <div className="flex-1 text-center">
                      <p className="text-xs text-gray-500 aileron-bold mb-1">
                        {t.Mobile_View?.Postpone || "Postpone"}
                      </p>
                      <div className="flex justify-center">
                        <label className="relative inline-block w-10 h-5 cursor-pointer">
                          <input
                            type="checkbox"
                            checked={campaign.status === "postpone"}
                            onChange={() => togglePostpone(campaign.id)}
                            className="hidden"
                          />
                          <div
                            className={`absolute inset-0 border-2 rounded-full transition-all duration-200 ${
                              campaign.status === "postpone"
                                ? "border-green-500 bg-green-100"
                                : "border-black bg-white"
                            }`}
                          ></div>
                          <div
                            className={`absolute top-1/2 -translate-y-1/2 w-3.5 h-3.5 rounded-full transition-all duration-200 ${
                              campaign.status === "postpone"
                                ? "left-[calc(100%-1.1rem)] bg-green-500"
                                : "left-[0.2rem] bg-black"
                            }`}
                          ></div>
                        </label>
                      </div>
                    </div>
                    <div className="flex-1 text-center">
                      <p className="text-xs text-gray-500 aileron-bold mb-1">
                        {t.Mobile_View?.Bill_Pdf || "Bill.pdf"}
                      </p>
                      <img
                        className="cursor-pointer w-6 h-6 mx-auto"
                        src="/blue-eye-icon.png"
                        alt={t.Alt_Text?.Invoice || "Invoice"}
                        onClick={() => handleDownloadInvoice(campaign)}
                      />
                    </div>
                  </div>

                  {/* Row 4: Refund and Notes */}
                  <div className="flex items-center justify-between pt-2">
                    <div className="flex-1 text-center">
                      <p className="text-xs text-gray-500 aileron-bold mb-1">
                        {t.Mobile_View?.Cost || "Cost€"}
                      </p>
                      <p className="avenir-roman text-sm font-semibold">
                        {campaign.payments?.[0]?.final_amount || "0"}
                      </p>
                    </div>
                    <div className="flex-1 text-center">
                      <p className="text-xs text-gray-500 aileron-bold mb-1">
                        {t.Mobile_View?.Refunded || "Refunded"}
                      </p>
                      <img
                        className="cursor-pointer w-6 h-6 mx-auto"
                        src="/red-circle.png"
                        alt={t.Alt_Text?.Refund || "Refund"}
                      />
                    </div>
                    <div className="flex-1 text-center">
                      <p className="text-xs text-gray-500 aileron-bold mb-1">
                        {t.Mobile_View?.Notes || "Notes"}
                      </p>
                      <img
                        className="cursor-pointer w-6 h-6 mx-auto"
                        src="/blue-eye-icon.png"
                        alt={t.Alt_Text?.Notes || "Notes"}
                      />
                    </div>
                  </div>
                </div>
              );
            })
          )}
        </div>
      </div>
    </>
  );
};

export default Adverts;
