import React, { ChangeEvent, InputHTMLAttributes } from 'react';
import "./label.css"
interface FormInputProps extends Omit<InputHTMLAttributes<HTMLInputElement>, 'onChange'> {
  label: string;
  name: string;
  value: string;
  onChange: (name: any, value: any) => void;
  isLoading?: boolean;
  required?: boolean;
  error?: string;
  labelType?: 'default' | 'optional';
  spacing?: 'none' | 'top' | 'top-two' | 'top-three';
  containerWidth?: 'full' | '70';
}

const FormInput: React.FC<FormInputProps> = ({
  label,
  name,
  value,
  onChange,
  isLoading = false,
  required = false,
  error,
  labelType = 'default',
  spacing = 'none',
  containerWidth = '70',
  className = '',
  ...inputProps
}) => {
  // Handle input change
  const handleChange = (e: ChangeEvent<HTMLInputElement>) => {
    onChange(name, e.target.value);
  };

  // Determine label class
  const getLabelClass = () => {
    const baseClass = labelType === 'optional' ? 'form-label-optional' : 'form-label';
    const spacingClass = spacing !== 'none' ? `form-label-spacetop-${spacing}` : '';
    return `${baseClass} ${spacingClass}`.trim();
  };

  // Determine input class
  const getInputClass = () => {
    const baseClass = 'form-input';
    const errorClass = error ? 'form-input-error' : '';
    return `${baseClass} ${errorClass} ${className}`.trim();
  };

  // Determine container class
  const getContainerClass = () => {
    const baseClass = 'form-input-group';
    const widthClass = containerWidth === 'full' ? 'w-full' : '';
    return `${baseClass} ${widthClass}`.trim();
  };

  return (
    <div className={getContainerClass()}>
      <label htmlFor={name} className={getLabelClass()}>
        {label}
        {required && labelType === 'default' && <span className="required-asterisk"></span>}
      </label>
      
      <input
        id={name}
        name={name}
        value={value}
        onChange={handleChange}
        disabled={isLoading}
        className={getInputClass()}
        required={required}
        {...inputProps}
      />
      
      {error && (
        <div className="error-message">
          {error}
        </div>
      )}
    </div>
  );
};

export default FormInput;