type DataType =
  | "sepData"
  | "cityHubData"
  | "categoryData"
  | "summaryData"
  | "sepTableData"
  | "eventHubTableData"
  | "categoryTableData"
  | "sidebarData"
  | "settingsData";

interface DataSources {
  sepData: () => Promise<any>;
  cityHubData: () => Promise<any>;
  categoryData: () => Promise<any>;
  summaryData: () => Promise<any>;
  sepTableData: () => Promise<any>;
  eventHubTableData: () => Promise<any>;
  categoryTableData: () => Promise<any>;
  sidebarData: () => Promise<any>;
  settingsData: () => Promise<any>;
}

const dataSources: DataSources = {
  sepData: () =>
    import("./selfserveportal.json").then((module) => module.default),
  cityHubData: () =>
    import("./cityhubpages.json").then((module) => module.default),
  categoryData: () =>
    import("./category.json").then((module) => module.default),
  summaryData: () => import("./summary.json").then((module) => module.default),
  sepTableData: () =>
    import("./sepTableJson.json").then((module) => module.default),
  eventHubTableData: () =>
    import("./eventHubTableJson.json").then((module) => module.default),
  categoryTableData: () =>
    import("./categoryTableJson.json").then((module) => module.default),
  sidebarData: () => import("./sidebar.json").then((module) => module.default),
  settingsData: () =>
    import("./settings.json").then((module) => module.default),
};

export const getData = async (dataType: DataType): Promise<any> => {
  if (!dataSources[dataType]) {
    throw new Error(
      `Data type '${dataType}' not found. Available types: ${Object.keys(
        dataSources
      ).join(", ")}`
    );
  }

  try {
    return await dataSources[dataType]();
  } catch (error) {
    console.error(`Error loading ${dataType}:`, error);
    throw new Error(`Failed to load ${dataType} data`);
  }
};

// Note: Auth-related data (login, register, forgot password, etc.) has been moved inline to components

// Individual exports
export const getSepData = (): Promise<any> => getData("sepData");
export const getCityHubData = (): Promise<any> => getData("cityHubData");
export const getCategoryData = (): Promise<any> => getData("categoryData");
export const getSummaryData = (): Promise<any> => getData("summaryData");
export const getSepTableData = (): Promise<any> => getData("sepTableData");
export const getEventHubTableData = (): Promise<any> =>
  getData("eventHubTableData");
export const getCategoryTableData = (): Promise<any> =>
  getData("categoryTableData");
export const getSidebarData = (): Promise<any> => getData("sidebarData");
export const getSettingsData = (): Promise<any> => getData("settingsData");

export default dataSources;
